// Copyright 2014 Google Inc. All Rights Reserved.

#include "AudioSink.h"

void AudioSink::addDiscoveryInfo(ServiceDiscoveryResponse* sdr) {
    Service* service = sdr->add_services();
    service->set_id(id());
    MediaSinkService* mss = service->mutable_media_sink_service();
    mss->set_available_type(mCodecType);
    mss->set_audio_type(mAudioType);

    for (vector<shared_ptr<AudioConfiguration> >::iterator it = mAudioConfigurations.begin();
            it != mAudioConfigurations.end(); ++it) {
        shared_ptr<AudioConfiguration> cur = *it;
        AudioConfiguration* ac = mss->add_audio_configs();
        ac->set_sampling_rate(cur->sampling_rate());
        ac->set_number_of_bits(cur->number_of_bits());
        ac->set_number_of_channels(cur->number_of_channels());
    }
}

void AudioSink::handleDataAvailable(uint64_t timestamp, const shared_ptr<IoBuffer>& data,
        size_t offset) {
    uint8_t* ptr = (uint8_t*) data->raw() + offset;
    size_t len = data->size() - offset;
    mCallbacks->dataAvailableCallback(mCurrentSessionId, timestamp, data, ptr, len);
}

int AudioSink::handleSetup(int mediaCodecType) {
    if (mCallbacks->setupCallback(mediaCodecType) != STATUS_SUCCESS) {
        return STATUS_MEDIA_CONFIG_MISMATCH;
    }
    sendConfig(Config_Status_STATUS_READY);
    return STATUS_SUCCESS;
}

int AudioSink::handleCodecConfig(void* config, size_t len) {
    return mCallbacks->codecConfigCallback((uint8_t*)config, len);
}

int AudioSink::handleMediaConfiguration(int index) {
    // TODO: Add a callback and pass this up to the sink.
    return STATUS_SUCCESS;
}

void AudioSink::playbackStart(int32_t sessionId) {
    mCallbacks->playbackStartCallback(sessionId);
}

void AudioSink::playbackStop(int32_t sessionId) {
    mCallbacks->playbackStopCallback(sessionId);
}

uint32_t AudioSink::addSupportedConfiguration(uint32_t samplingRate, uint32_t numberOfBits,
        uint32_t numberOfChannels) {
    shared_ptr<AudioConfiguration> ac(new AudioConfiguration());
    ac->set_sampling_rate(samplingRate);
    ac->set_number_of_bits(numberOfBits);
    ac->set_number_of_channels(numberOfChannels);
    mAudioConfigurations.push_back(ac);
    return mAudioConfigurations.size() - 1;
}

unsigned int AudioSink::getNumberOfConfigurations() {
    return mAudioConfigurations.size();
}

